#### Code for Paper: “On the Role of Minimum Variances in Weighted Optimal Designs”
## This R file provides the functions needed to run 
#  examples in "Weighted_Ex.R"

### Function to generate rank-restricted c-optimal designs
# N, sample size, scalar, any positive real number
# h, contrast of interest
# returns reps for c-optimal design
# that allows for estimation of all effects

approx_c_CRD <- function(N, h) {
  
  nt = length(h)
  if(N <= 0) return("N must be a positive, real number ") 

  # count number of nonzero treatments per contrast
  tH = sum(h != 0)
  
  scale = N - nt + tH
  rvec = rep(NA, nt)
  
  for(i in 1:nt) {
    if(h[i] == 0) { 
      rvec[i] = 1
    } else {
      rvec[i] = scale*abs(h[i])/sum(abs(h))
    }
  }
  
  return(rvec)
}

### Functions for Completely Randomized Design (CRD)
## Weighted_A 
# Returns the value of the weighted A objective function
# Inputs
# wi, vector of weights for treatment i
# rvec, vector of replications
Weighted_A = function(wi, rvec) {
  sum(wi/rvec)
}

### Function for Exact CRDs----
# exact_A_CRD
# Output rvec, number of replications for exact A_w optimal CRD
# rvec consists of positive integers
# Inputs
# N, scalar, number of runs
# wvec, vector of weights of length ns (number of contrasts)
# H, matrix of contrasts. Each row is a contrast of length number of treatments

exact_A_CRD <- function(N, wvec, H) {
  
  nt = ncol(H) # number of treatments
  ns = nrow(H) # number of contrasts
  
  if(N < nt) return("N must be greater than or equal to t") 
  if(length(wvec) != ns) return("Weight vector must be of length ns")
  # create vector of replication numbers
  # all start as 1 to ensure estimability
  rvec = rep(1, length = nt)
  
  wi_star = apply(H, 2, function(x) sum(wvec * x^2))
  
  # if saturated case, skip to the end
  
  if(N != nt) {
    
    Ntemp    = sum(rvec) # how many cases left to run?
    
    # remove columns with all 0 values
    # since they will always have r = 1
    # (treatment is involved in no contrasts)
    ind_zero = which(apply(H, 2, function(x) all(x == 0)) == TRUE)
    if(length(ind_zero) == 0) {
      rn = nt
    } else {
      rvec  = rvec[-ind_zero]
      rn    = length(rvec)
    }
    while(Ntemp < N) {
      add1      = t(replicate(rn, rvec)) + diag(rn)
      cof       = apply(add1, 1, function(x) Weighted_A(wi_star, x) )
      ind       = which.min(cof) # find rep w/ smallest value obj fun
      rvec[ind] = rvec[ind] + 1 # add rep to appropriate trt
      Ntemp     = Ntemp + 1 # increase iter counter by 1
    } # end while loop
    
    # add back in cases where contrast coef is 0
    rvec[ind_zero] = 1 
  } # end of if loop
  
  # output results
  cof           = Weighted_A(wi_star, rvec)
  result        = list(rvec, cof)
  names(result) = c("Reps", "ObjFunValue")
  return(result)
}

## var_fun
# Output: produces the variance of all contrasts
# Input
# rvec, vector of replications, from exact_A_CRD
# H, matrix of contrasts. Each row is a contrast of length number of treatments
var_fun = function(rvec, H) {
  var = apply(H, 1, function(x) sum((x^2)/rvec))
  return(var)
}